﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using VeteransAffairs.Registries.BusinessAHOBPR.Extensions;
using MI = VeteransAffairs.Registries.BusinessAHOBPR.MilitaryInfoService;
using VeteransAffairs.Registries.BusinessAHOBPR;
using VeteransAffairs.Registries.BusinessAHOBPR.PaymentService;

namespace VeteransAffairs.Registries.BusinessManagerAHOBPR.Emis
{
    public class RegistrantDeploymentFactory : DeploymentAbstractFactory
    {
        private Dictionary<string, int> stdComponent;
        private Dictionary<string, string> countryNamesByCode;
        public RegistrantDeploymentFactory() { }
        public RegistrantDeploymentFactory(string dbConnInfo) : base(dbConnInfo) { }
        protected override IDeployment Create(DoDDeploymentAdapter deployment)
        {
            return new REGISTRANT_DEPLOYMENT().Create(deployment);
        }
        public override List<IDeployment> CreateDeployments(MI.eMISdeploymentResponseType deploymentResponse, RegistrantHistory history) //Dictionary<DateRange, string> branchesOfService)// Implementation of Factory Method.
        {
            List<IDeployment> deployments = new List<IDeployment>();


            foreach (MI.Deployment emisDeployment in deploymentResponse.deployment)
            {
                if (emisDeployment != null)
                {
                    AddDeploymentsToList(deployments, history, emisDeployment);
                }
            }
            return deployments;
        }
        public override List<IDeployment> CreateDeployments(MI.eMISdeploymentResponseType deploymentResponse, Dictionary<DateRange, string> branchesOfService)
        {
            var edipi = deploymentResponse?.deployment?.First()?.edipi;
            var history = new RegistrantHistory();
            history.OccupationsByDateTime = new MilitaryOccupationHistoryManager(connectionInfo).GetRegistrantHistory(edipi);
            history.PaymentDataByDateTime = new PaymentHistoryManager().GetRegistrantHistory(edipi);
            history.BranchesOfService = branchesOfService;
            return CreateDeployments(deploymentResponse, history);
        }


        private void AddDeploymentsToList(List<IDeployment> deployments, RegistrantHistory history, MI.Deployment emisDeployment)
        {
            if (history == null) throw new ArgumentNullException(nameof(history));
            if (emisDeployment.deploymentData.DeploymentLocation != null)
            {
                CreateCountryNamesByCode();
                foreach (MI.DeploymentLocation location in emisDeployment.deploymentData.DeploymentLocation)
                {
                    var deploymentAdapter = CreateAdapter(emisDeployment, location, history);
                    REGISTRANT_DEPLOYMENT deployment = Create(deploymentAdapter) as REGISTRANT_DEPLOYMENT;
                    deployments.Add(deployment);
                }
            }
        }
        private void CreateCountryNamesByCode()
        {
            if (countryNamesByCode == null)
            {
                using (var db = GetLocalContext(connectionInfo))
                {
                    db.DeferredLoadingEnabled = false;
                    countryNamesByCode = db.STD_COUNTRies.ToDictionary(x => x.CODE, x => x.DESCRIPTION);
                }
            }
        }
        private DoDDeploymentAdapter CreateAdapter(MI.Deployment emisDeployment, MI.DeploymentLocation location, RegistrantHistory history)
        {
            DateRange deploymentsByDateRange = CreateNewDateRange(location.deploymentLocationBeginDate, location.deploymentLocationEndDate);
            BranchOfService branchOfService = new BranchOfService(location.deploymentLocationBeginDate,
                                                    location.deploymentLocationEndDate, history.BranchesOfService);
            var deployment = new DoDDeploymentAdapter();
            deployment.PersonnelCategoryCode = GetStdComponent(emisDeployment?.keyData?.personnelCategoryTypeCode?.Trim()).ToString();
            deployment.DeploymentStageId = (int)Enums.DeploymentStage.NotSet;
            deployment.BranchOfService = branchOfService;
            //A deploymentlocation will either have a country OR a major body of water code
            SetDeploymentCountry(location, deployment);
            SetDeploymentDates(deploymentsByDateRange, deployment);
            SetEligibilities(deploymentsByDateRange, deployment);
            SetDeploymentsandPayPlan(history, deploymentsByDateRange, deployment);
            //Console.WriteLine($"Leaving adapter: DepOccupations '{deployment.ServiceSpecificOccupation.Count}");
            return deployment;
        }

        private void SetDeploymentCountry(MI.DeploymentLocation location, DoDDeploymentAdapter deployment)
        {
            var key = location?.deploymentCountry?.Trim() ?? location.deploymentLocationMajorBodyOfWaterCode?.Trim();

            deployment.DeploymentCountry = countryNamesByCode.ContainsKey(key) ? countryNamesByCode[key] : key;
        }

        private void SetDeploymentsandPayPlan(RegistrantHistory history, DateRange deploymentsByDateRange, DoDDeploymentAdapter deployment)
        {
            var mapper = new RegistrantHistoryToDeploymentMapper();
            SetDeploymentServiceSpecificOccupations(mapper, history, deploymentsByDateRange, deployment);
            SetPayPlan(mapper, history, deploymentsByDateRange, deployment);
        }

        private void SetPayPlan(RegistrantHistoryToDeploymentMapper mapper, RegistrantHistory history, DateRange deploymentsByDateRange, DoDDeploymentAdapter deployment)
        {
            var payGradeData = mapper.GetPayGradeHistoryForDateRange(history, deploymentsByDateRange);
            var payPlan = new PaymentHistoryManager(connectionInfo).LookupPayPlan(payGradeData, deployment.BranchOfService.Code);
            if (payPlan != null)
            {
                deployment.PayPlanId = payPlan.STD_PAYPLAN_ID;
                deployment.MilitaryRank = payPlan.RANK_TEXT;
            }
        }
        private void SetDeploymentServiceSpecificOccupations(RegistrantHistoryToDeploymentMapper mapper, RegistrantHistory history, DateRange deploymentsByDateRange, DoDDeploymentAdapter deployment)
        {
            deployment.ServiceSpecificOccupation = mapper.SetDeploymentOccupations(history, deploymentsByDateRange);
        }

        private void SetEligibilities(DateRange deploymentDateRange, DoDDeploymentAdapter deployment)
        {
            deployment.IsEligible = CheckForEligibility(EligibilityValidatorFactory.CreateAHOBPREligibilityValidator(), deploymentDateRange, deployment.DeploymentCountry);
            deployment.IsGulfWarVeteran = CheckForEligibility(EligibilityValidatorFactory.CreateFirstGulfWarValidator(), deploymentDateRange, deployment.DeploymentCountry);
        }
        private static void SetDeploymentDates(DateRange deploymentDateRange, DoDDeploymentAdapter deployment)
        {
            deployment.DeploymentStartDate = deploymentDateRange.Start;
            deployment.DeploymentEndDate = deploymentDateRange.End;
        }
        private bool CheckForEligibility(EligibilityValidator validator, DateRange deploymentDateRange, string deploymentCountry)
        {
            return validator.IsEligible(deploymentDateRange, deploymentCountry);
        }

        private Dictionary<string, int> StdComponent
        {
            get
            {
                if (stdComponent == null)
                {
                    stdComponent = new Dictionary<string, int>
                    {
                        {"a"                , (int)Enums.Compo.ActiveDuty},
                        {"activeduty"       , (int)Enums.Compo.ActiveDuty},
                        {"v"                , (int)Enums.Compo.GuardReserve},
                        {"guardreserve"     , (int)Enums.Compo.GuardReserve},
                        {"n"                , (int)Enums.Compo.NationalGuard},
                        {"nationalguard"    , (int)Enums.Compo.NationalGuard},
                        {"r"                , (int)Enums.Compo.Retiree},
                        {"retiree"          , (int)Enums.Compo.Retiree},
                        {"q"                , (int)Enums.Compo.ReserveRetiree},
                        {"reserveretiree"   , (int)Enums.Compo.ReserveRetiree}
                    };
                }
                return stdComponent;
            }
        }
        private int GetStdComponent(string component)
        {
            return (!string.IsNullOrEmpty(component) && StdComponent.ContainsKey(component.ToLower())) ? StdComponent[component.ToLower()] : (int)Enums.Compo.NoDataAvailable;
        }


    }
}
